/*	Copyright 2002 by Eric Postpischil, http://edp.org.
	See license information in index.html.
*/


typedef	char	Bool;	// smaller Boolean than Microsoft's BOOL


#include	"Shapes.h"


// Specify values of customizable application settings.
#define	STANDARD_SIZE		50		// pixel length of triangle side at "100%"
#define	INTERIORPEN			FALSE	// outline interior triangles
#define	PERIMETERPENCOLOR	(RGB(255, 255, 255))
									// color for borders of pieces
#define	PERIMETERPENPIXELS	1		// thickness for borders of pieces
#define	SCROLL_SPEED		20		// milliseconds between smooth scroll updates

// Define submenu positions.
#define	POS_FILE_MENU	0
#define	POS_PIECES_MENU	1
#define	POS_GOAL_MENU	2
#define	POS_VIEW_MENU	3


/*	There are a number of places where we must divide a length
	of pixels into two nearly equal pieces.  When the length, n,
	is odd, the two pieces must be different sizes.  One could
	be n/2, and then the other must be n-n/2.  The latter is
	equivalent to (n+1)/2, which is preferable since it does
	not evaluate n twice.
*/
#define	SMALLER_HALF(n)	((n)/2)
#define	LARGER_HALF(n)	(((n)+1)/2)


/*	Define a structure for program options.  This is passed to a dialog
	box, used in the main window, and recorded in the Windows registry.
	Changes from version to version must be compatible!
*/
typedef struct {
		unsigned int	interiorPen : 1;			// Boolean for lines around triangles
													// Could add other bits later.
		unsigned char	perimeterPenPixels;			// thickness of lines around shapes
		COLORREF		perimeterPenColor;			// color of lines around shapes
}	OptionsInfo;


// Global functions.
ATOM	TriDisplayRegister(HINSTANCE hInstance),
		StatusRegister(HINSTANCE hInstance);
void	DisplayError(const char *title, const char *message),
		DisplayCError(const char *title),
		DisplayWindowsError(const char *title);
void	freeShape(Shape *shape),
		freeShapeList(ShapeList *list),
		freePieces(Pieces *shapes);
Shape	*readShape(FILE *file),
		*copyShape(const Shape *shape),
		*aggregatePieces(Pieces *pieces, int width, int height),
		*sampleShape(void),
		*newShape(void);
Pieces	*extractPieces(Shape *shape, Bool extractP);
Bool	writeShape(FILE *file, const Shape *shape),
		compressShape(Shape *shape);
Point	toggleTriangle(Shape *shape, int x, int y);
void	Sobol(int num, COLORREF *color);
void	print(HWND window, const OptionsInfo *options, const Shape *shape),
		printShape(HDC hdc, const OptionsInfo *options, Shape *shape, RECT clip);

BOOL CALLBACK	About(HWND dialog, UINT message, WPARAM wParam, LPARAM lParam);
BOOL CALLBACK	Options(HWND dialog, UINT message, WPARAM wParam, LPARAM lParam);
BOOL CALLBACK	Status(HWND dialog, UINT message, WPARAM wParam, LPARAM lParam);
BOOL CALLBACK	ZoomTo(HWND dialog, UINT message, WPARAM wParam, LPARAM lParam);
void			*StartSolver(HWND window, Pieces *pieces, ShapeList *work);
void			StopSolver(void *);


/*	_beginthread is defined to return an unsigned long and is
	specified to return -1 upon failure.  That's poor documentation,
	but the value is presumably (unsigned long) -1.  Define a symbol
	for that.
*/
#define	NO_THREAD	((unsigned long) -1)


//	Define the parameters for the TriDisplay window.

// The following structure gives scrolling and mapping information for one dimension.
typedef	struct {
		SCROLLINFO		scroll;			// Windows scroll bar information
						// (See comments about info.nTrackPos in TriDisplay.cpp.)
		short			posMax;			// maximum value of nPos
		short			length;			// length of this side of client rectangle
		short			velocity;		// smooth scroll current velocity
		short			bar;			// scroll bar ID for ScrollBarInfo
		short			indexToPage;	// index-space to page-space scale
		short			extentDevice;	// viewport extent
}	ScrollInfo;

// Now the above two types are assembled with others into the TriDisplayInfo type.
typedef struct {
		// Private to window.
		Shape			*shape;						// shape to display
		ScrollInfo		horz, vert;					// scroll bar information
		short			lineIncrement;				// amount to move for line scroll
		OptionsInfo		options;					// display options
		unsigned		perimeterPenThickness;		// page space thickness
		unsigned short	viewSize;					// pixels for length of triangle side
		char			viewMode;					// scaling method
		DWORD			lastScroll;					// time of last smooth scroll update
		Bool			editMode;					// editing is enabled
		HCURSOR			cursor;						// editing cursor
		Point			lastEditIndex,				// index-space location of last edit
						lastEditDevice;				// device-space location of last edit
		Bool			captured;					// mouse is captured
}	TriDisplayInfo;


// Define the parameters for the Options dialog.
typedef struct {
		// Passed in.
		OptionsInfo		options;					// options to set and return

		// Private.
		TriDisplayInfo	sampleInfo;					// space for sample display window
		HWND			sampleWindow;				// sample window handle
}	OptionsDialogInfo;


//	Define the parameters for the Status dialog box.
typedef struct {
		// Passed in by creator.
		WORD		ID;				// child window ID
		ShapeList	work;			// Solver work and output list header

		// Private to window.
		int			numCurrent,		// number of currently displayed solution
					numSolutions;	// number of solutions found so far
		ShapeList	*current;		// currently displayed solution
		HBITMAP		solverOn,		// bitmap of turned-on switch
					solverOff;		// bitmap of turned-off switch
}	StatusInfo;


// Message flags (see below).
enum {
		CE_IMPERATIVE	= 1,
		CE_REDRAW		= 2,
		CE_EDIT			= 4,
};				// exhibit change flags
enum { VIEW_FIT, VIEW_FIXED, VIEW_ZOOMIN, VIEW_ZOOMOUT };
														// view modes and commands
enum { SOLVER_SUCCESS, SOLVER_ERROR, SOLVER_ABORTED };	// Solver returns


/*	Application-defined messages.
	We start at WM_USER+2 because dialog boxes use two "user" codes.
*/
enum {
	WM_SOLVER_STARTING = WM_USER+2,
	/*	Sent by the poser to the status dialog before starting
		Solver.  Status dialog should free any prior Solver work.
	*/

	WM_SOLUTION_FOUND,
	/*	Sent by the Solver thread to the problem poser when it
		finds a new solution.  No parameters, the new solution
		is in a list known to the poser.  Also forwarded to
		the status dialog.
	*/

	WM_SOLVER_ENDING,
	/*	Sent by the Solver to say it is ending.  Also forwarded
		to the status dialog.  wParam contains the result status.
	*/

	WM_SETSHAPE,
	/*	Command to display a shape or notice that a displayed shape
		has changed.

		HIWORD(wParam)		Bit mask:
			CE_IMPERATIVE	Change is a command that must be obeyed.
							In the absence of this flag, the message is
							a notice that the shape has been changed and
							should be updated if it is currently displayed.
			CE_REDRAW		Redraw background.  This is necessary when the
							outline of the shape changes.  When only the
							contents of the shape change, as when the Solver
							is moving pieces around inside the goal, this
							flag can and should be omitted.  Not redrawing
							the background reduces screen flicker.
			CE_EDIT			When sent to a display window, puts it in edit
							mode.  Guide lines are drawn, an edit cursor is
							used, and changes to the shape may be mode.
							When sent from a display window, informs
							parent that an edit has been made.

		lParam				Pointer to new shape.
	*/

	WM_GETSHAPE,
	/*	Ask a window to report shape it is displaying.
		(Shape **) lParam	address at which to store address of shape
	*/

	WM_SETVIEW,
	/*	Tell a window to set a new view.
		wParam			Enum:
			VIEW_FIT	fit to window
			VIEW_FIXED	scale triangles to given size
		(unsigned short) lParam
						size in pixels of a triangle edge, if
						VIEW_FIXED specified (must be non-zero)
	*/

	WM_SETOPTIONS,
	/*	Tell a window the application options have changed.
		(Options *) lParam
						pointer to new options
	*/

	WM_SHOWMESSAGE,
	// Tell status window to show a message.
};